"""
Reflex Web Scraper & LLM Analyzer
A visual framework for scraping websites and analyzing content with LLMs
"""

import reflex as rx
from typing import List, Dict, Optional
from datetime import datetime
import random
import logging
import os
import json
import csv
import io
import asyncio
from concurrent.futures import ThreadPoolExecutor
import requests
from bs4 import BeautifulSoup
from urllib.parse import urljoin, urlparse
from dotenv import load_dotenv

# Local imports
from .llm_operations import extract_website, run_llm, get_qdrant_client

# Load environment variables
load_dotenv()

# Configuration
LOCAL_FILES_DIR = os.getenv('LOCAL_FILES_DIR') if os.environ.get('LOCAL_FILES_DIR') is not None else './local_files'
COMPANIES_FILE = LOCAL_FILES_DIR + '/companies.json'
ANALYSES_FILE = LOCAL_FILES_DIR + '/analyses.json'

QUESTIONS_FILE = 'proprietary/questions.json'
CREDENTIALS_FILE = 'proprietary/credentials.json'

ADMIN_USER = os.getenv('ADMIN_USER')
ADMIN_PASSWORD = os.getenv('ADMIN_PASSWORD')

# Setup logging
logging.basicConfig(
    format='%(asctime)s %(levelname)s:%(message)s',
    level=logging.INFO
)

# Get Qdrant client from llm_operations module
client = get_qdrant_client()

# Thread pool for running blocking operations
executor = ThreadPoolExecutor(max_workers=4)


# Questions loading function
def load_questions() -> List[Dict[str, str]]:
    """Load questions from JSON file."""
    try:
        if os.path.exists(QUESTIONS_FILE):
            with open(QUESTIONS_FILE, 'r', encoding='utf-8') as f:
                questions_data = json.load(f)
                return questions_data
        else:
            logging.warning(f"Questions file not found: {QUESTIONS_FILE}")
            return []
    except Exception as e:
        logging.error(f"Error loading questions from {QUESTIONS_FILE}: {str(e)}")
        return []


# Company persistence functions
def load_companies() -> Dict[str, str]:
    """Load companies from JSON file."""
    try:
        if os.path.exists(COMPANIES_FILE):
            with open(COMPANIES_FILE, 'r', encoding='utf-8') as f:
                return json.load(f)
        else:
            # Return empty dict if file doesn't exist
            return {}
    except Exception as e:
        logging.error(f"Error loading companies from {COMPANIES_FILE}: {str(e)}")
        return {}


def save_companies(companies: Dict[str, str]):
    """Save companies to JSON file."""
    try:
        with open(COMPANIES_FILE, 'w', encoding='utf-8') as f:
            json.dump(companies, f, indent=4, ensure_ascii=False)
    except Exception as e:
        logging.error(f"Error saving companies to {COMPANIES_FILE}: {str(e)}")


# Analysis persistence functions
def load_analyses() -> Dict:
    """Load analysis history and related data from JSON file."""
    try:
        if os.path.exists(ANALYSES_FILE):
            with open(ANALYSES_FILE, 'r', encoding='utf-8') as f:
                data = json.load(f)
                analysis_history = data.get('analysis_history', {})

                # Migrate old format: parse JSON strings in answer field
                for company_name, analyses in analysis_history.items():
                    for analysis in analyses:
                        migrated = False
                        ja_count = 0

                        for result in analysis.get('results', []):
                            answer = result.get('answer', '')

                            # Check if answer field contains unparsed JSON
                            if answer and not result.get('description') and not result.get('source_name'):
                                try:
                                    cleaned_answer = answer.strip()

                                    # Try to parse as JSON
                                    answer_json = json.loads(cleaned_answer)

                                    # Update result with parsed fields
                                    result['question'] = answer_json.get('question', result.get('question', ''))
                                    result['answer'] = answer_json.get('answer', '')
                                    result['description'] = answer_json.get('description', '')
                                    result['source_name'] = answer_json.get('source_name', '')
                                    result['source_url'] = answer_json.get('source_url', '')

                                    migrated = True
                                except (json.JSONDecodeError, TypeError):
                                    # Not JSON, leave as-is
                                    pass

                            # Count "Ja" answers for this analysis
                            if result.get('answer', '').strip().lower() == 'ja':
                                ja_count += 1

                        # Update items_found for this analysis if migrated
                        if migrated:
                            analysis['items_found'] = ja_count
                            logging.info(f"Migrated JSON responses for {company_name}, found {ja_count} 'Ja' answers")

                # Rebuild items_found dict from latest analysis of each company
                items_found = {}
                for company_name, analyses in analysis_history.items():
                    if analyses:
                        # Get the most recent analysis
                        latest_analysis = analyses[-1]
                        items_found[company_name] = latest_analysis.get('items_found', 0)

                return {
                    'analysis_history': analysis_history,
                    'items_found': items_found,
                    'last_analyzed': data.get('last_analyzed', {})
                }
        else:
            return {
                'analysis_history': {},
                'items_found': {},
                'last_analyzed': {}
            }
    except Exception as e:
        logging.error(f"Error loading analyses from {ANALYSES_FILE}: {str(e)}")
        return {
            'analysis_history': {},
            'items_found': {},
            'last_analyzed': {}
        }


def save_analyses(analysis_history: Dict, items_found: Dict, last_analyzed: Dict):
    """Save analysis history and related data to JSON file."""
    try:
        data = {
            'analysis_history': analysis_history,
            'items_found': items_found,
            'last_analyzed': last_analyzed
        }
        with open(ANALYSES_FILE, 'w', encoding='utf-8') as f:
            json.dump(data, f, indent=4, ensure_ascii=False)
    except Exception as e:
        logging.error(f"Error saving analyses to {ANALYSES_FILE}: {str(e)}")


def verify_credentials(username: str, password: str) -> bool:
    if username == ADMIN_USER and password == ADMIN_PASSWORD:
        return True
    else:
        return False


class WebsiteAnalyzerState(rx.State):
    """State management for the website analyzer app."""

    # Authentication state
    is_authenticated: bool = False
    login_username: str = ""
    login_password: str = ""
    login_error: str = ""

    # Website management - Dictionary of company name -> URL (loaded from file)
    companies: Dict[str, str] = load_companies()

    # Load persisted analysis data
    _loaded_analyses = load_analyses()

    # Tracking for scan and analysis timestamps
    last_scanned: Dict[str, str] = {}
    last_analyzed: Dict[str, str] = _loaded_analyses['last_analyzed']

    # Items found count per company (from latest analysis)
    items_found: Dict[str, int] = _loaded_analyses['items_found']

    # Analysis history: company_name -> list of analyses with timestamps
    analysis_history: Dict[str, List[Dict]] = _loaded_analyses['analysis_history']

    # Currently processing company
    scraping_company: str = ""
    analyzing_company: str = ""

    # Progress tracking
    scan_progress: str = ""
    analysis_progress: str = ""

    # Multi-select functionality
    selected_companies: List[str] = []
    bulk_operation_in_progress: bool = False
    bulk_operation_status: str = ""

    # Questions to be answered (loaded from file)
    questions: List[Dict[str, str]] = load_questions()

    # Analysis results (dummy data for now)
    analysis_results: List[Dict[str, str]] = []

    # UI State
    selected_company: str = ""
    selected_analysis_index: int = 0
    current_analysis_timestamp: str = ""

    # Configuration state
    edit_company_name: str = ""
    edit_company_url: str = ""
    editing_company_key: str = ""  # Original key when editing existing company

    def get_qdrant_last_scan_time(self, company_name: str) -> str:
        """Get the last scan timestamp from Qdrant collection metadata."""
        try:
            if client.collection_exists(collection_name=company_name):
                collection_info = client.get_collection(collection_name=company_name)
                # Get the points count to verify collection has data
                if collection_info.points_count > 0:
                    # Retrieve a single point to get the scan timestamp from metadata
                    scroll_result = client.scroll(
                        collection_name=company_name,
                        limit=1,
                        with_payload=True,
                        with_vectors=False
                    )
                    if scroll_result and scroll_result[0] and len(scroll_result[0]) > 0:
                        point = scroll_result[0][0]
                        # Extract scan_timestamp from payload metadata
                        if point.payload and 'metadata' in point.payload:
                            metadata = point.payload['metadata']
                            if 'scan_timestamp' in metadata:
                                return metadata['scan_timestamp']

                    # Fallback to manually tracked timestamp if not found in Qdrant
                    return self.last_scanned.get(company_name, "Scanned (date unknown)")
                else:
                    return "Never"
            else:
                return "Never"
        except Exception as e:
            logging.error(f"Error getting Qdrant collection info for {company_name}: {str(e)}")
            return self.last_scanned.get(company_name, "Never")

    @rx.var
    def company_list(self) -> List[Dict[str, str]]:
        """Get companies as a list of dicts for easier iteration."""
        result = []
        for name, url in self.companies.items():
            items = self.items_found.get(name, 0)
            # Determine color based on items found
            if items >= 5:
                color = "green"
            elif items >= 3:
                color = "orange"
            elif items > 0:
                color = "red"
            else:
                color = "gray"

            # Check if analysis has been run
            has_been_analyzed = name in self.analysis_history and len(self.analysis_history[name]) > 0

            # Get last scanned time from Qdrant or fallback to tracked time
            last_scanned = self.get_qdrant_last_scan_time(name)

            result.append({
                "name": name,
                "url": url,
                "last_scanned": last_scanned,
                "last_analyzed": self.last_analyzed.get(name, "Never"),
                "items_found": items,
                "items_found_display": f"{items}/7" if has_been_analyzed else "-",
                "badge_color": color,
                "has_been_analyzed": has_been_analyzed,
            })
        return result

    @rx.var
    def available_analyses(self) -> List[str]:
        """Get list of available analysis timestamps for selected company."""
        if not self.selected_company or self.selected_company not in self.analysis_history:
            return []
        return [
            analysis["timestamp"]
            for analysis in self.analysis_history[self.selected_company]
        ]

    @rx.var
    def analysis_count(self) -> int:
        """Get count of analyses for selected company."""
        if not self.selected_company or self.selected_company not in self.analysis_history:
            return 0
        return len(self.analysis_history[self.selected_company])

    async def scrape_website(self, company_name: str):
        """Scrape website and extract content to vector database."""
        self.scraping_company = company_name
        self.scan_progress = "Scan wordt gestart..."
        yield  # Force UI update to show loading state
        url = self.companies.get(company_name, "")

        if not url:
            yield rx.toast.error(f"Geen URL gevonden voor {company_name}")
            self.scraping_company = ""
            self.scan_progress = ""
            return

        try:
            loop = asyncio.get_event_loop()

            # Use extract_website from llm_operations module
            self.scan_progress = "Website-inhoud wordt geëxtraheerd..."
            yield

            await loop.run_in_executor(executor, extract_website, company_name, url)

            # Update timestamp
            self.last_scanned[company_name] = datetime.now().strftime("%Y-%m-%d %H:%M")

            self.scan_progress = "Scan succesvol afgerond!"
            yield
            await asyncio.sleep(0.5)

            yield rx.toast.success(f"Scan van {company_name} succesvol afgerond")
        except Exception as e:
            logging.exception(f"Error scanning {company_name}: {str(e)}")
            self.scan_progress = f"Fout: {str(e)}"
            yield
            yield rx.toast.error(f"Fout bij scannen van {company_name}: {str(e)}")
        finally:
            self.scraping_company = ""
            self.scan_progress = ""

    async def analyze_website(self, company_name: str):
        """Analyze website using LLM based on predefined questions."""
        self.analyzing_company = company_name
        self.selected_company = company_name
        self.analysis_progress = "Analyse wordt gestart..."
        yield  # Force UI update to show loading state

        # Check if the website has been scanned
        if not client.collection_exists(collection_name=company_name):
            yield rx.toast.error(f"Geen websitegegevens gevonden voor {company_name}. Scan eerst de website.")
            self.analyzing_company = ""
            self.analysis_progress = ""
            return

        try:
            # Generate timestamp
            timestamp = datetime.now().strftime("%Y-%m-%d %H:%M:%S")
            self.current_analysis_timestamp = timestamp

            # Run LLM analysis for each question
            results = []
            found_count = 0
            loop = asyncio.get_event_loop()

            for i, question_obj in enumerate(self.questions, 1):
                self.analysis_progress = f"Analyseren {i}/{len(self.questions)}..."
                yield

                question_text = question_obj['question']
                clarification = question_obj.get('clarification', '')
                logging.info(f"Analyzing {i}: {question_text}")

                # Run LLM in thread pool to keep WebSocket alive
                llm_result = await loop.run_in_executor(executor, run_llm, company_name, question_text, clarification)
                answer = llm_result['answer']

                # Parse JSON response from LLM
                try:
                    cleaned_answer = answer.strip()

                    # Try to parse the answer as JSON
                    answer_json = json.loads(cleaned_answer)

                    # Count how many questions were answered with "Ja"
                    if answer_json.get('answer', '').strip().lower() == 'ja':
                        found_count += 1

                    results.append({
                        "company": company_name,
                        "question_number": str(i),
                        "question": answer_json.get('question', question_text),
                        "answer": answer_json.get('answer', ''),
                        "description": answer_json.get('description', ''),
                        "source_name": answer_json.get('source_name', ''),
                        "source_url": answer_json.get('source_url', '')
                    })
                except json.JSONDecodeError:
                    # Fallback for non-JSON responses (backward compatibility)
                    logging.warning(f"Could not parse JSON response for question {i}, using raw answer")

                    # Count how many questions were answered with "Ja" (old format)
                    if answer.strip().lower().startswith('ja'):
                        found_count += 1

                    results.append({
                        "company": company_name,
                        "question_number": str(i),
                        "question": question_text,
                        "answer": answer,
                        "description": "",
                        "source_name": "",
                        "source_url": ""
                    })

                # Yield to keep WebSocket connection alive
                await asyncio.sleep(0)

            # Store found count
            self.items_found[company_name] = found_count

            # Store in history
            if company_name not in self.analysis_history:
                self.analysis_history[company_name] = []

            self.analysis_history[company_name].append({
                "timestamp": timestamp,
                "results": results,
                "items_found": found_count
            })

            # Set current results to latest analysis
            self.analysis_results = results
            self.selected_analysis_index = len(self.analysis_history[company_name]) - 1

            # Update last analyzed timestamp
            self.last_analyzed[company_name] = datetime.now().strftime("%Y-%m-%d %H:%M")

            # Save analyses to file
            save_analyses(self.analysis_history, self.items_found, self.last_analyzed)

            yield rx.toast.success(f"Analyse afgerond voor {company_name}")
        except Exception as e:
            logging.exception(f"Error analyzing {company_name}: {str(e)}")
            yield rx.toast.error(f"Fout bij analyseren van {company_name}: {str(e)}")
        finally:
            self.analyzing_company = ""

        # Redirect to results page
        yield rx.redirect("/results")

    def view_analysis(self, company_name: str, analysis_index: int):
        """View a specific analysis from history."""
        if company_name in self.analysis_history:
            analyses = self.analysis_history[company_name]
            if 0 <= analysis_index < len(analyses):
                self.selected_company = company_name
                self.selected_analysis_index = analysis_index
                self.analysis_results = analyses[analysis_index]["results"]
                self.current_analysis_timestamp = analyses[analysis_index]["timestamp"]
                # Update items_found to match the viewed analysis
                self.items_found[company_name] = analyses[analysis_index].get("items_found", 0)
                return rx.redirect("/results")

    def view_latest_analysis(self, company_name: str):
        """View the most recent analysis from history."""
        if company_name in self.analysis_history:
            analyses = self.analysis_history[company_name]
            if analyses:
                # View the last (most recent) analysis
                latest_index = len(analyses) - 1
                return self.view_analysis(company_name, latest_index)

    def select_analysis(self, timestamp: str):
        """Select an analysis by timestamp."""
        if self.selected_company in self.analysis_history:
            for idx, analysis in enumerate(self.analysis_history[self.selected_company]):
                if analysis["timestamp"] == timestamp:
                    self.selected_analysis_index = idx
                    self.analysis_results = analysis["results"]
                    self.current_analysis_timestamp = timestamp
                    # Update items_found to match the selected analysis
                    self.items_found[self.selected_company] = analysis.get("items_found", 0)
                    break

    def clear_results(self):
        """Clear all analysis results."""
        self.analysis_results = []
        self.selected_company = ""
        self.current_analysis_timestamp = ""
        return rx.redirect("/")

    # Multi-select methods
    def toggle_company_selection(self, company_name: str):
        """Toggle selection of a company."""
        if company_name in self.selected_companies:
            self.selected_companies.remove(company_name)
        else:
            self.selected_companies.append(company_name)

    def select_all_companies(self):
        """Select all companies."""
        self.selected_companies = list(self.companies.keys())

    def clear_selection(self):
        """Clear all selections."""
        self.selected_companies = []

    def toggle_select_all(self, checked: bool):
        """Toggle select all companies based on checkbox state."""
        if checked:
            self.selected_companies = list(self.companies.keys())
        else:
            self.selected_companies = []

    async def bulk_scan(self):
        """Scan all selected companies."""
        if not self.selected_companies:
            yield rx.toast.error("Geen bedrijven geselecteerd")
            return

        # Copy the list before clearing selection
        companies_to_process = list(self.selected_companies)
        total = len(companies_to_process)

        self.bulk_operation_in_progress = True
        self.bulk_operation_status = f"Scannen 0/{total} bedrijven..."
        yield

        for idx, company_name in enumerate(companies_to_process, 1):
            self.bulk_operation_status = f"Scannen {idx}/{total}: {company_name}"
            yield

            # Call the existing scrape_website method
            async for _ in self.scrape_website(company_name):
                yield

        self.bulk_operation_in_progress = False
        self.bulk_operation_status = ""
        self.clear_selection()
        yield rx.toast.success(f"Scannen van {total} bedrijven afgerond")

    async def bulk_analyze(self):
        """Analyze all selected companies."""
        if not self.selected_companies:
            yield rx.toast.error("Geen bedrijven geselecteerd")
            return

        # Copy the list before clearing selection
        companies_to_process = list(self.selected_companies)
        total = len(companies_to_process)

        self.bulk_operation_in_progress = True
        self.bulk_operation_status = f"Analyseren 0/{total} bedrijven..."
        yield

        for idx, company_name in enumerate(companies_to_process, 1):
            self.bulk_operation_status = f"Analyseren {idx}/{total}: {company_name}"
            yield

            # Call the existing analyze_website method
            async for _ in self.analyze_website(company_name):
                yield

        self.bulk_operation_in_progress = False
        self.bulk_operation_status = ""
        self.clear_selection()
        yield rx.toast.success(f"Analyseren van {total} bedrijven afgerond")

    async def bulk_scan_and_analyze(self):
        """Scan and analyze all selected companies."""
        if not self.selected_companies:
            yield rx.toast.error("Geen bedrijven geselecteerd")
            return

        # Copy the list before clearing selection
        companies_to_process = list(self.selected_companies)
        total = len(companies_to_process)

        self.bulk_operation_in_progress = True
        yield

        for idx, company_name in enumerate(companies_to_process, 1):
            # Scan first
            self.bulk_operation_status = f"Scannen {idx}/{total}: {company_name}"
            yield

            async for _ in self.scrape_website(company_name):
                yield

            # Then analyze
            self.bulk_operation_status = f"Analyseren {idx}/{total}: {company_name}"
            yield

            async for _ in self.analyze_website(company_name):
                yield

        self.bulk_operation_in_progress = False
        self.bulk_operation_status = ""
        self.clear_selection()
        yield rx.toast.success(f"Scannen en analyseren van {total} bedrijven afgerond")

    # Company management methods
    def set_edit_company_name(self, value: str):
        """Update the company name field."""
        self.edit_company_name = value

    def set_edit_company_url(self, value: str):
        """Update the company URL field."""
        self.edit_company_url = value

    def start_edit_company(self, company_name: str):
        """Start editing an existing company."""
        self.editing_company_key = company_name
        self.edit_company_name = company_name
        self.edit_company_url = self.companies.get(company_name, "")

    def clear_company_form(self):
        """Clear the company edit form."""
        self.edit_company_name = ""
        self.edit_company_url = ""
        self.editing_company_key = ""

    def add_or_update_company(self):
        """Add a new company or update an existing one."""
        if not self.edit_company_name or not self.edit_company_url:
            return rx.toast.error("Bedrijfsnaam en URL zijn verplicht")

        # If editing an existing company and name changed, remove old entry
        if self.editing_company_key and self.editing_company_key != self.edit_company_name:
            if self.editing_company_key in self.companies:
                del self.companies[self.editing_company_key]
                # Transfer history if exists
                if self.editing_company_key in self.analysis_history:
                    self.analysis_history[self.edit_company_name] = self.analysis_history.pop(self.editing_company_key)
                if self.editing_company_key in self.last_scanned:
                    self.last_scanned[self.edit_company_name] = self.last_scanned.pop(self.editing_company_key)
                if self.editing_company_key in self.last_analyzed:
                    self.last_analyzed[self.edit_company_name] = self.last_analyzed.pop(self.editing_company_key)
                if self.editing_company_key in self.items_found:
                    self.items_found[self.edit_company_name] = self.items_found.pop(self.editing_company_key)

        # Add or update company
        self.companies[self.edit_company_name] = self.edit_company_url

        # Store company name before clearing form
        saved_company_name = self.edit_company_name

        # Save to file
        save_companies(self.companies)

        # Clear form
        self.clear_company_form()

        return rx.toast.success(f"Bedrijf '{saved_company_name}' succesvol opgeslagen")

    def delete_company(self, company_name: str):
        """Delete a company from the list."""
        if company_name in self.companies:
            del self.companies[company_name]

            # Save to file
            save_companies(self.companies)

            # Clean up related data
            if company_name in self.analysis_history:
                del self.analysis_history[company_name]
            if company_name in self.last_scanned:
                del self.last_scanned[company_name]
            if company_name in self.last_analyzed:
                del self.last_analyzed[company_name]
            if company_name in self.items_found:
                del self.items_found[company_name]

            # Save analyses to file
            save_analyses(self.analysis_history, self.items_found, self.last_analyzed)

            return rx.toast.success(f"Bedrijf '{company_name}' succesvol verwijderd")
        return rx.toast.error(f"Bedrijf '{company_name}' niet gevonden")

    def handle_login(self):
        """Handle user login."""
        if not self.login_username or not self.login_password:
            self.login_error = "Voer gebruikersnaam en wachtwoord in"
            return

        if verify_credentials(self.login_username, self.login_password):
            self.is_authenticated = True
            self.login_error = ""
            self.login_password = ""  # Clear password from state

            # Reload all data from files to ensure fresh state
            self.reload_analysis_data()
            self.companies = load_companies()

            return rx.redirect("/")
        else:
            self.login_error = "Ongeldige gebruikersnaam of wachtwoord"
            self.login_password = ""  # Clear password from state

    def handle_login_keydown(self, key):
        """Handle keydown on login form - submit on Enter."""
        if key == "Enter":
            return self.handle_login()

    def handle_logout(self):
        """Handle user logout."""
        self.is_authenticated = False
        self.login_username = ""
        self.login_password = ""
        self.login_error = ""
        return rx.redirect("/login")

    def check_login(self):
        """Check if user is authenticated, redirect to login if not."""
        if not self.is_authenticated:
            return rx.redirect("/login")
        else:
            # Reload analysis data when re-authenticated to get latest timestamps
            self.reload_analysis_data()

    def reload_analysis_data(self):
        """Reload analysis data from file to ensure we have the latest information."""
        loaded_analyses = load_analyses()
        self.last_analyzed = loaded_analyses['last_analyzed']
        self.items_found = loaded_analyses['items_found']
        self.analysis_history = loaded_analyses['analysis_history']

    def copy_analysis_to_clipboard(self):
        """Format the current analysis results for copying to clipboard."""
        if not self.analysis_results or not self.selected_company:
            return rx.toast.error("Geen analyseresultaten om te kopiëren")

        # Build formatted text for email
        lines = []
        lines.append(f"Analyse resultaten voor: {self.selected_company}")
        if self.current_analysis_timestamp:
            lines.append(f"Analysedatum: {self.current_analysis_timestamp}")
        lines.append("")
        lines.append("=" * 60)
        lines.append("")

        for result in self.analysis_results:
            # Question
            lines.append(result.get("question", ""))
            lines.append("")

            # Answer
            answer = result.get("answer", "")
            if answer:
                lines.append(f"Antwoord: {answer}")
                lines.append("")

            # Description
            description = result.get("description", "")
            if description:
                lines.append(description)
                lines.append("")

            # Source
            source_name = result.get("source_name", "")
            source_url = result.get("source_url", "")
            if source_name:
                if source_url:
                    lines.append(f"Bron: {source_name} ({source_url})")
                else:
                    lines.append(f"Bron: {source_name}")
                lines.append("")

            lines.append("-" * 60)
            lines.append("")

        formatted_text = "\n".join(lines)

        # Use rx.set_clipboard to copy to clipboard
        return [
            rx.set_clipboard(formatted_text),
            rx.toast.success("Analyse gekopieerd naar klembord")
        ]

    def export_to_csv(self):
        """Export companies table to CSV file."""
        if not self.companies:
            return rx.toast.error("Geen bedrijven om te exporteren")

        # Create CSV in memory
        output = io.StringIO()
        writer = csv.writer(output, delimiter=';')

        # Write header matching the main table
        writer.writerow([
            "Bedrijf",
            "URL",
            "Laatst gescand",
            "Laatst geanalyseerd",
            "Items gevonden"
        ])

        # Write data for each company
        for company in self.company_list:
            items = company["items_found"]
            writer.writerow([
                company["name"],
                company["url"],
                company["last_scanned"],
                company["last_analyzed"],
                items  # Just the number, not "X/7" format
            ])

        # Get CSV content
        csv_content = output.getvalue()
        output.close()

        # Generate filename with timestamp
        filename = f"companies_{datetime.now().strftime('%Y%m%d_%H%M%S')}.csv"

        # Trigger download
        return rx.download(data=csv_content.encode('utf-8'), filename=filename)


def company_row(company: Dict[str, str]) -> rx.Component:
    """Create a single company row for the table."""
    company_name = company["name"]

    return rx.table.row(
        rx.table.cell(
            rx.checkbox(
                checked=WebsiteAnalyzerState.selected_companies.contains(company_name),
                on_change=WebsiteAnalyzerState.toggle_company_selection(company_name),
                disabled=WebsiteAnalyzerState.bulk_operation_in_progress,
            )
        ),
        rx.table.cell(
            rx.link(
                company["name"],
                href=company["url"],
                font_weight="bold",
                color="blue",
                text_decoration="none",
                _hover={"text_decoration": "underline"},
                is_external=True,
            )
        ),
        rx.table.cell(
            rx.text(company["last_scanned"], size="2")
        ),
        rx.table.cell(
            rx.flex(
                rx.text(company["last_analyzed"], size="2"),
                rx.cond(
                    WebsiteAnalyzerState.analysis_history.contains(company_name),
                    rx.icon_button(
                        rx.icon("eye", size=14),
                        on_click=WebsiteAnalyzerState.view_latest_analysis(company_name),
                        size="1",
                        color_scheme="gray",
                        variant="ghost",
                    ),
                ),
                spacing="2",
                align="center",
            )
        ),
        rx.table.cell(
            rx.cond(
                company["has_been_analyzed"],
                rx.badge(
                    company["items_found_display"],
                    color_scheme=company["badge_color"],
                    size="2",
                    cursor="pointer",
                    _hover={"opacity": "0.8"},
                    on_click=WebsiteAnalyzerState.view_latest_analysis(company_name),
                ),
                rx.text("-", size="2", color="gray.400"),
            )
        ),
        rx.table.cell(
            rx.vstack(
                rx.flex(
                    rx.button(
                        rx.icon("download", size=16),
                        "Scannen",
                        on_click=WebsiteAnalyzerState.scrape_website(company_name),
                        loading=WebsiteAnalyzerState.scraping_company == company_name,
                        disabled=(WebsiteAnalyzerState.scraping_company != "") | (WebsiteAnalyzerState.analyzing_company != ""),
                        size="2",
                        color_scheme="blue",
                        variant="soft",
                    ),
                    rx.button(
                        rx.icon("brain", size=16),
                        "Analyseren",
                        on_click=WebsiteAnalyzerState.analyze_website(company_name),
                        loading=WebsiteAnalyzerState.analyzing_company == company_name,
                        disabled=(WebsiteAnalyzerState.scraping_company != "") | (WebsiteAnalyzerState.analyzing_company != ""),
                        size="2",
                        color_scheme="green",
                        variant="soft",
                    ),
                    spacing="2",
                ),
                rx.cond(
                    (WebsiteAnalyzerState.scraping_company == company_name) & (WebsiteAnalyzerState.scan_progress != ""),
                    rx.text(
                        WebsiteAnalyzerState.scan_progress,
                        size="1",
                        color="blue.600",
                        font_style="italic",
                    ),
                ),
                rx.cond(
                    (WebsiteAnalyzerState.analyzing_company == company_name) & (WebsiteAnalyzerState.analysis_progress != ""),
                    rx.text(
                        WebsiteAnalyzerState.analysis_progress,
                        size="1",
                        color="green.600",
                        font_style="italic",
                    ),
                ),
                spacing="1",
                align="start",
                width="100%",
            )
        ),
    )


def companies_table() -> rx.Component:
    """Component displaying companies in a table with scan/analyze actions."""
    return rx.box(
        rx.table.root(
            rx.table.header(
                rx.table.row(
                    rx.table.column_header_cell(
                        rx.checkbox(
                            checked=WebsiteAnalyzerState.selected_companies.length() == WebsiteAnalyzerState.company_list.length(),
                            on_change=WebsiteAnalyzerState.toggle_select_all,
                            disabled=WebsiteAnalyzerState.bulk_operation_in_progress,
                        ),
                        width="5%"
                    ),
                    rx.table.column_header_cell("Bedrijf", width="27%"),
                    rx.table.column_header_cell("Laatst gescand", width="16%"),
                    rx.table.column_header_cell("Laatst geanalyseerd", width="16%"),
                    rx.table.column_header_cell("Items gevonden", width="12%"),
                    rx.table.column_header_cell("Acties", width="24%"),
                )
            ),
            rx.table.body(
                rx.foreach(
                    WebsiteAnalyzerState.company_list,
                    company_row,
                )
            ),
            variant="surface",
            size="3",
        ),
        width="100%",
    )


def results_overview() -> rx.Component:
    """Component displaying analysis results in an overview table."""
    return rx.box(
        rx.flex(
            rx.vstack(
                rx.heading("Analyseresultaten", size="6"),
                rx.cond(
                    WebsiteAnalyzerState.selected_company != "",
                    rx.flex(
                        rx.text(
                            f"Bedrijf: {WebsiteAnalyzerState.selected_company}",
                            size="2",
                            font_weight="medium",
                        ),
                        rx.cond(
                            WebsiteAnalyzerState.current_analysis_timestamp != "",
                            rx.text(
                                f" | Analyse van: {WebsiteAnalyzerState.current_analysis_timestamp}",
                                size="2",
                                color="gray.600",
                            ),
                        ),
                        spacing="1",
                        align="center",
                    ),
                ),
                spacing="2",
                align="start",
            ),
            rx.flex(
                rx.cond(
                    WebsiteAnalyzerState.analysis_count > 1,
                    rx.select.root(
                        rx.select.trigger(
                            placeholder="Bekijk eerdere analyse...",
                        ),
                        rx.select.content(
                            rx.foreach(
                                WebsiteAnalyzerState.available_analyses,
                                lambda ts: rx.select.item(ts, value=ts),
                            )
                        ),
                        value=WebsiteAnalyzerState.current_analysis_timestamp,
                        on_change=WebsiteAnalyzerState.select_analysis,
                        size="2",
                    ),
                ),
                rx.button(
                    rx.icon("copy", size=16),
                    "Kopieer analyse",
                    on_click=WebsiteAnalyzerState.copy_analysis_to_clipboard,
                    size="2",
                    color_scheme="blue",
                    variant="soft",
                ),
                rx.button(
                    "Wis resultaten",
                    on_click=WebsiteAnalyzerState.clear_results,
                    size="2",
                    color_scheme="red",
                    variant="soft",
                ),
                spacing="2",
            ),
            justify="between",
            align="start",
            margin_bottom="1rem",
        ),
        rx.cond(
            WebsiteAnalyzerState.analysis_results.length() > 0,
            rx.box(
                rx.foreach(
                    WebsiteAnalyzerState.analysis_results,
                    lambda result: rx.box(
                        # Question header
                        rx.text(
                            result["question"],
                            font_weight="bold",
                            size="4",
                            margin_bottom="0.75rem",
                        ),
                        # Answer badge
                        rx.cond(
                            result["answer"] != "",
                            rx.badge(
                                result["answer"],
                                color_scheme=rx.cond(
                                    result["answer"].lower() == "ja",
                                    "green",
                                    "red"
                                ),
                                size="2",
                                margin_bottom="0.75rem",
                            ),
                        ),
                        # Description
                        rx.cond(
                            result["description"] != "",
                            rx.box(
                                rx.markdown(
                                    result["description"],
                                    component_map={
                                        "p": lambda text: rx.text(text, size="3", as_="p"),
                                        "ul": lambda text: rx.list.unordered(text, size="3"),
                                        "ol": lambda text: rx.list.ordered(text, size="3"),
                                        "li": lambda text: rx.list.item(text, size="3"),
                                        "a": lambda text: rx.link(text, is_external=True),
                                    },
                                    white_space="pre-wrap",
                                ),
                                margin_bottom="0.75rem",
                            ),
                        ),
                        # Source information
                        rx.cond(
                            result["source_name"] != "",
                            rx.box(
                                rx.text(
                                    "Bron:",
                                    size="2",
                                    font_weight="medium",
                                    color="gray.600",
                                    margin_bottom="0.25rem",
                                ),
                                rx.cond(
                                    result["source_url"] != "",
                                    rx.link(
                                        result["source_name"],
                                        href=result["source_url"],
                                        color="blue",
                                        text_decoration="underline",
                                        is_external=True,
                                        size="2",
                                    ),
                                    rx.text(
                                        result["source_name"],
                                        size="2",
                                        color="gray.700",
                                    ),
                                ),
                                padding_top="0.5rem",
                                border_top="1px solid",
                                border_color="gray.200",
                            ),
                        ),
                        padding="1.25rem",
                        border_radius="0.5rem",
                        border="1px solid",
                        border_color="gray.200",
                        background_color="white",
                        margin_bottom="1rem",
                    )
                ),
            ),
            rx.callout(
                "Nog geen analyseresultaten. Selecteer een website en klik op 'Analyseren' om hier resultaten te zien.",
                icon="info",
                color_scheme="gray",
            ),
        ),
        padding="1.5rem",
        border_radius="0.75rem",
        border="1px solid",
        border_color="gray.200",
        background_color="gray.50",
    )


def login_page() -> rx.Component:
    """Login page component."""
    return rx.container(
        rx.center(
            rx.card(
                rx.vstack(
                    rx.vstack(
                        rx.text(
                            "Gebruikersnaam",
                            size="2",
                            font_weight="medium",
                            margin_bottom="0.25rem",
                        ),
                        rx.input(
                            placeholder="Voer gebruikersnaam in",
                            value=WebsiteAnalyzerState.login_username,
                            on_change=WebsiteAnalyzerState.set_login_username,
                            size="3",
                            width="100%",
                        ),
                        width="100%",
                        spacing="1",
                    ),
                    rx.vstack(
                        rx.text(
                            "Wachtwoord",
                            size="2",
                            font_weight="medium",
                            margin_bottom="0.25rem",
                        ),
                        rx.input(
                            type="password",
                            placeholder="Voer wachtwoord in",
                            value=WebsiteAnalyzerState.login_password,
                            on_change=WebsiteAnalyzerState.set_login_password,
                            on_key_down=WebsiteAnalyzerState.handle_login_keydown,
                            size="3",
                            width="100%",
                        ),
                        width="100%",
                        spacing="1",
                    ),
                    rx.cond(
                        WebsiteAnalyzerState.login_error != "",
                        rx.callout(
                            WebsiteAnalyzerState.login_error,
                            icon="triangle_alert",
                            color_scheme="red",
                            size="1",
                        ),
                    ),
                    rx.button(
                        "Inloggen",
                        on_click=WebsiteAnalyzerState.handle_login,
                        size="3",
                        width="100%",
                        color_scheme="blue",
                    ),
                    spacing="4",
                    width="100%",
                ),
                size="4",
                max_width="400px",
            ),
            height="100vh",
        ),
        padding="2rem",
    )


def index() -> rx.Component:
    """Main page component."""
    return rx.container(
        rx.vstack(
            # Header
            rx.box(
                rx.flex(
                    rx.flex(
                        rx.link(
                            rx.box(
                                rx.image(
                                    src="/logo.png",
                                    height="40px",
                                    width="auto",
                                ),
                                background="white",
                                padding="0.3rem 0.6rem",
                                border_radius="0.25rem",
                                cursor="pointer",
                                _hover={"opacity": "0.9"},
                            ),
                            href="/",
                        ),
                        rx.heading(
                            "Deelnemersanalyse",
                            size="6",
                            color="white",
                            margin_left="1rem",
                        ),
                        align="center",
                        spacing="3",
                    ),
                    rx.button(
                        rx.icon("log-out", size=16),
                        "Uitloggen",
                        on_click=WebsiteAnalyzerState.handle_logout,
                        size="2",
                        variant="soft",
                        color="white",
                    ),
                    justify="between",
                    align="center",
                    width="100%",
                ),
                background="linear-gradient(135deg, #1d1e40 0%, #1e7f94 100%)",
                padding="1rem 1.5rem",
                border_radius="0.5rem",
                margin_bottom="0.5rem",
                width="100%",
            ),

            # Bulk action buttons (always visible)
            rx.box(
                rx.flex(
                    rx.cond(
                        WebsiteAnalyzerState.selected_companies.length() > 0,
                        rx.text(
                            f"{WebsiteAnalyzerState.selected_companies.length()} bedrijf/bedrijven geselecteerd",
                            size="2",
                            font_weight="medium",
                            color="gray.700",
                        ),
                        rx.text(
                            "Geen bedrijven geselecteerd",
                            size="2",
                            color="gray.500",
                        ),
                    ),
                    rx.spacer(),
                    rx.button(
                        rx.icon("download", size=16),
                        "Bulk scannen",
                        on_click=WebsiteAnalyzerState.bulk_scan,
                        loading=WebsiteAnalyzerState.bulk_operation_in_progress,
                        disabled=(WebsiteAnalyzerState.selected_companies.length() == 0) | WebsiteAnalyzerState.bulk_operation_in_progress,
                        size="2",
                        color_scheme="blue",
                        variant="soft",
                    ),
                    rx.button(
                        rx.icon("brain", size=16),
                        "Bulk analyseren",
                        on_click=WebsiteAnalyzerState.bulk_analyze,
                        loading=WebsiteAnalyzerState.bulk_operation_in_progress,
                        disabled=(WebsiteAnalyzerState.selected_companies.length() == 0) | WebsiteAnalyzerState.bulk_operation_in_progress,
                        size="2",
                        color_scheme="green",
                        variant="soft",
                    ),
                    rx.button(
                        rx.icon("zap", size=16),
                        "Scannen + Analyseren",
                        on_click=WebsiteAnalyzerState.bulk_scan_and_analyze,
                        loading=WebsiteAnalyzerState.bulk_operation_in_progress,
                        disabled=(WebsiteAnalyzerState.selected_companies.length() == 0) | WebsiteAnalyzerState.bulk_operation_in_progress,
                        size="2",
                        color_scheme="purple",
                        variant="soft",
                    ),
                    rx.button(
                        "Selectie wissen",
                        on_click=WebsiteAnalyzerState.clear_selection,
                        disabled=(WebsiteAnalyzerState.selected_companies.length() == 0) | WebsiteAnalyzerState.bulk_operation_in_progress,
                        size="2",
                        variant="outline",
                    ),
                    rx.divider(orientation="vertical", size="2"),
                    rx.icon_button(
                        rx.icon("download", size=20),
                        on_click=WebsiteAnalyzerState.export_to_csv,
                        size="2",
                        color_scheme="gray",
                        variant="ghost",
                        title="Exporteer analyseresultaten naar CSV",
                    ),
                    rx.link(
                        rx.icon_button(
                            rx.icon("settings", size=20),
                            size="2",
                            color_scheme="gray",
                            variant="ghost",
                            title="Configuratie",
                        ),
                        href="/config",
                    ),
                    spacing="3",
                    align="center",
                    width="100%",
                ),
                rx.cond(
                    WebsiteAnalyzerState.bulk_operation_status != "",
                    rx.text(
                        WebsiteAnalyzerState.bulk_operation_status,
                        size="2",
                        color="blue.600",
                        font_style="italic",
                        margin_top="0.5rem",
                    ),
                ),
                padding="1rem",
                border_radius="0.5rem",
                border="1px solid",
                border_color="blue.200",
                background_color="blue.50",
                margin_bottom="0.5rem",
                width="100%",
            ),

            # Companies table (full width)
            companies_table(),

            spacing="4",
            width="100%",
        ),
        size="4",
        padding_y="2rem",
    )


def results_page() -> rx.Component:
    """Results page component."""
    return rx.container(
        rx.vstack(
            # Header
            rx.box(
                rx.flex(
                    rx.flex(
                        rx.link(
                            rx.box(
                                rx.image(
                                    src="/logo-cso.png",
                                    height="40px",
                                    width="auto",
                                ),
                                background="white",
                                padding="0.3rem 0.6rem",
                                border_radius="0.25rem",
                                cursor="pointer",
                                _hover={"opacity": "0.9"},
                            ),
                            href="/",
                        ),
                        rx.heading(
                            "Deelnemersanalyse",
                            size="6",
                            color="white",
                            margin_left="1rem",
                        ),
                        align="center",
                        spacing="3",
                    ),
                    rx.button(
                        rx.icon("log-out", size=16),
                        "Uitloggen",
                        on_click=WebsiteAnalyzerState.handle_logout,
                        size="2",
                        variant="soft",
                        color="white",
                    ),
                    justify="between",
                    align="center",
                    width="100%",
                ),
                background="linear-gradient(135deg, #1d1e40 0%, #1e7f94 100%)",
                padding="1rem 1.5rem",
                border_radius="0.5rem",
                margin_bottom="0.5rem",
                width="100%",
            ),

            # Back button - prominent placement
            rx.link(
                rx.button(
                    rx.icon("arrow-left", size=20),
                    "Terug naar bedrijven",
                    size="3",
                    color_scheme="blue",
                    variant="solid",
                ),
                href="/",
            ),

            # Results section (full width)
            results_overview(),

            spacing="6",
            width="100%",
        ),
        size="4",
        padding_y="2rem",
    )


def config_page() -> rx.Component:
    """Configuration page component."""
    return rx.container(
        rx.vstack(
            # Header
            rx.box(
                rx.flex(
                    rx.flex(
                        rx.link(
                            rx.box(
                                rx.image(
                                    src="/logo-cso.png",
                                    height="40px",
                                    width="auto",
                                ),
                                background="white",
                                padding="0.3rem 0.6rem",
                                border_radius="0.25rem",
                                cursor="pointer",
                                _hover={"opacity": "0.9"},
                            ),
                            href="/",
                        ),
                        rx.heading(
                            "Deelnemersanalyse",
                            size="6",
                            color="white",
                            margin_left="1rem",
                        ),
                        align="center",
                        spacing="3",
                    ),
                    rx.button(
                        rx.icon("log-out", size=16),
                        "Uitloggen",
                        on_click=WebsiteAnalyzerState.handle_logout,
                        size="2",
                        variant="soft",
                        color="white",
                    ),
                    justify="between",
                    align="center",
                    width="100%",
                ),
                background="linear-gradient(135deg, #1d1e40 0%, #1e7f94 100%)",
                padding="1rem 1.5rem",
                border_radius="0.5rem",
                margin_bottom="0.5rem",
                width="100%",
            ),

            # Back button
            rx.link(
                rx.button(
                    rx.icon("arrow-left", size=20),
                    "Terug naar bedrijven",
                    size="3",
                    color_scheme="blue",
                    variant="solid",
                ),
                href="/",
            ),

            # Configuration section
            rx.box(
                rx.heading("Bedrijfsconfiguratie", size="6", margin_bottom="1rem"),

                # Add/Edit Form
                rx.box(
                    rx.heading(
                        rx.cond(
                            WebsiteAnalyzerState.editing_company_key != "",
                            "Bedrijf bewerken",
                            "Nieuw bedrijf toevoegen"
                        ),
                        size="4",
                        margin_bottom="1rem"
                    ),
                    rx.vstack(
                        rx.box(
                            rx.text("Bedrijfsnaam", size="2", font_weight="medium", margin_bottom="0.5rem"),
                            rx.input(
                                placeholder="Voer bedrijfsnaam in",
                                value=WebsiteAnalyzerState.edit_company_name,
                                on_change=WebsiteAnalyzerState.set_edit_company_name,
                                size="3",
                                width="100%",
                            ),
                            width="100%",
                        ),
                        rx.box(
                            rx.text("Website URL", size="2", font_weight="medium", margin_bottom="0.5rem"),
                            rx.input(
                                placeholder="https://voorbeeld.nl",
                                value=WebsiteAnalyzerState.edit_company_url,
                                on_change=WebsiteAnalyzerState.set_edit_company_url,
                                size="3",
                                width="100%",
                            ),
                            width="100%",
                        ),
                        rx.flex(
                            rx.button(
                                rx.icon("save", size=16),
                                rx.cond(
                                    WebsiteAnalyzerState.editing_company_key != "",
                                    "Bedrijf bijwerken",
                                    "Bedrijf toevoegen"
                                ),
                                on_click=WebsiteAnalyzerState.add_or_update_company,
                                size="3",
                                color_scheme="green",
                            ),
                            rx.cond(
                                WebsiteAnalyzerState.editing_company_key != "",
                                rx.button(
                                    "Annuleren",
                                    on_click=WebsiteAnalyzerState.clear_company_form,
                                    size="3",
                                    color_scheme="gray",
                                    variant="soft",
                                ),
                            ),
                            spacing="2",
                        ),
                        spacing="4",
                        width="100%",
                    ),
                    padding="1.5rem",
                    border_radius="0.5rem",
                    border="1px solid",
                    border_color="blue.200",
                    background_color="blue.50",
                    margin_bottom="2rem",
                ),

                # Companies List
                rx.heading("Bestaande bedrijven", size="4", margin_bottom="1rem"),
                rx.table.root(
                    rx.table.header(
                        rx.table.row(
                            rx.table.column_header_cell("Bedrijfsnaam", width="30%"),
                            rx.table.column_header_cell("Website URL", width="50%"),
                            rx.table.column_header_cell("Acties", width="20%"),
                        )
                    ),
                    rx.table.body(
                        rx.foreach(
                            WebsiteAnalyzerState.company_list,
                            lambda company: rx.table.row(
                                rx.table.cell(
                                    rx.text(company["name"], font_weight="medium")
                                ),
                                rx.table.cell(
                                    rx.link(
                                        company["url"],
                                        href=company["url"],
                                        size="2",
                                        color="blue",
                                        is_external=True,
                                    )
                                ),
                                rx.table.cell(
                                    rx.flex(
                                        rx.button(
                                            rx.icon("pencil", size=14),
                                            on_click=WebsiteAnalyzerState.start_edit_company(company["name"]),
                                            size="2",
                                            color_scheme="blue",
                                            variant="soft",
                                        ),
                                        rx.button(
                                            rx.icon("trash", size=14),
                                            on_click=WebsiteAnalyzerState.delete_company(company["name"]),
                                            size="2",
                                            color_scheme="red",
                                            variant="soft",
                                        ),
                                        spacing="2",
                                    )
                                ),
                            )
                        )
                    ),
                    variant="surface",
                    size="3",
                ),
                width="100%",
            ),

            spacing="6",
            width="100%",
        ),
        size="4",
        padding_y="2rem",
    )


def require_login(page_func):
    """Decorator to protect pages that require authentication."""
    def wrapper() -> rx.Component:
        return rx.fragment(
            rx.cond(
                WebsiteAnalyzerState.is_authenticated,
                page_func(),
                rx.center(
                    rx.spinner(size="3"),
                    height="100vh",
                ),
            ),
        )
    return wrapper


# App configuration
app = rx.App(
    theme=rx.theme(
        appearance="light",
        accent_color="blue",
    )
)
app.add_page(login_page, route="/login", title="Inloggen")
app.add_page(require_login(index), route="/", title="Deelnemersanalyse", on_load=WebsiteAnalyzerState.check_login)
app.add_page(require_login(results_page), route="/results", title="Analyseresultaten", on_load=WebsiteAnalyzerState.check_login)
app.add_page(require_login(config_page), route="/config", title="Configuratie", on_load=WebsiteAnalyzerState.check_login)
